ZomboidSaver = {}
ZomboidSaver.__index = ZomboidSaver

function ZomboidSaver:new(o)
    o = o or {}
    setmetatable(o, self)
    o.__index = self.__index
    return o
end

function ZomboidSaver:save_server_config(server_object, server_name)
    local server_table = {}
    for k, v in pairs(server_object) do
        local write_out = tostring(v)
        if string.len(k) > 0 and string.sub(k, 1, 2) ~= '__' then
            server_table[#server_table + 1] = k .. '=' .. write_out
        end
    end
    local output_filename = ZomboidDeployInterface.schema.ExportDirectory.value
    output_filename = path_append(output_filename, server_name .. '_server.ini')
    local write_file = false
    if file_exists(output_filename) then
        write_file = confirm('Do you want to overwrite the file ' .. output_filename .. '?')
    else
        write_file = true
    end
    if write_file then
        local output_file = io.open(output_filename, 'w')
        output_file:write(table.concat(server_table, '\n'))
        output_file:close()
    end
    return write_file
end

function ZomboidSaver:save_sandbox_config(sandbox_object, server_name)
    local sandbox_table = sandbox_object.settings
    local output_filename = ZomboidDeployInterface.schema.ExportDirectory.value
    output_filename = path_append(output_filename, server_name .. '_SandboxVars.lua')
    local next_object = {
        nil,
        sandbox_table
    }
    local write_file = false
    if file_exists(output_filename) then
        write_file = confirm('Do you want to overwrite the file ' .. output_filename .. '?')
    else
        write_file = true
    end
    if write_file == false then
        return false
    end
    local output_handle = io.open(output_filename, 'w')
    local hierarchy_stack = {}
    local stack_restore = {}
    local indent_level = 0
    while next_object ~= nil do
        local key = next_object[1]
        local value = next_object[2]
        next_object = hierarchy_stack[1]
        table.remove(hierarchy_stack, 1)
        local prepend = ''
        for i = 1, indent_level do
            for j = 1, 4 do
                prepend = prepend .. ' '
            end
        end
        if type(value) == 'string' then
            value = '\'' .. value .. '\''
        end
        if type(value) == 'table' then
            table.insert(stack_restore, 1, {next_object, hierarchy_stack})
            hierarchy_stack = {}
            table.insert(hierarchy_stack, 1, {nil, nil})
            for k, v in pairs(value) do
                table.insert(hierarchy_stack, 1, {k, v})
            end
            next_object = hierarchy_stack[1]
            table.remove(hierarchy_stack, 1)
            if key ~= nil then
                output_handle:write(prepend .. key .. ' = ' .. '{\n')
            elseif key == nil then
                output_handle:write('{\n')
            end
            indent_level = indent_level + 1
        elseif key == nil and value == nil then
            local stack_pop = stack_restore[1]
            table.remove(stack_restore, 1)
            next_object = stack_pop[1]
            hierarchy_stack = stack_pop[2]
            indent_level = indent_level - 1
            if indent_level < 0 then
                indent_level = 0
            end
            prepend = ''
            for i = 1, indent_level do
                for j = 1, 4 do
                    prepend = prepend .. ' '
                end
            end
            if next_object == nil then
                output_handle:write('}\n')
            else
                output_handle:write(prepend .. '},\n')
            end
        else
            output_handle:write(prepend .. key .. ' = ' .. tostring(value) .. ',\n')
        end
    end
    output_handle:close()
    return true
end

ZomboidDeployInterface =
{
    display_order = {
        'ExportDirectory',
        'ServerName',
        'ExportServer',
        'ExportSandbox',
    },
    schema =
    {
        ExportDirectory =
        {
            description = 'Export Directory',
            type = DirectoryBrowse,
            value = application_path(),
            default = application_path(),
            setter = function(directory)
                if directory_exists(directory) then
                    ZomboidDeployInterface.schema.ExportDirectory.value = directory
                end
            end,
            getter = function() return ZomboidDeployInterface.schema.ExportDirectory.value end
        },
        ServerName =
        {
            description = 'Server Name',
            default = 'MyServer',
            type = String,
            value = 'MyServer',
            setter = function(v)
                ZomboidDeployInterface.schema.ServerName.value = v
            end,
            getter = function()
                return ZomboidDeployInterface.schema.ServerName.value
            end
        },
        ExportServer =
        {
            description = 'Export Server Config',
            type = Button,
            action = function()
                local saver = ZomboidSaver:new()
                if ZomboidServerInterface == nil or ZomboidServerInterface.instance == nil then
                    message('It does not appear that the Zomboid Server plugin is loaded.')
                    return
                end
                local server_object = ZomboidServerInterface.instance
                local success = saver:save_server_config(server_object, ZomboidDeployInterface.schema.ServerName.getter())
                if success then
                    message('Server Config Exported Successfully.')
                end
                saver = nil
            end,
            getter = function() return 'Export Server Config' end            
        },
        ExportSandbox =
        {
            description = 'Export Sandbox Config',
            type = Button,
            action = function()
                local saver = ZomboidSaver:new()
                if ZomboidSandboxVarsInterface == nil or ZomboidSandboxVarsInterface.instance == nil then
                    message('It does not appear that the Zomboid Sandbox plugin is loaded.')
                    return
                end
                local server_object = ZomboidSandboxVarsInterface.instance
                local success = saver:save_sandbox_config(server_object, ZomboidDeployInterface.schema.ServerName.getter())
                if success then
                    message('Sandbox Config Exported Successfully.')
                end
                saver = nil
            end,
            getter = function() return 'Export Sandbox Config' end            
        }
    }
}

register_plugin('Zomboid', 'Deploy\\/Export', ZomboidDeployInterface)